/*:
 * @target MZ
 * @plugindesc Enforces global volume levels for BGM, BGS, SE, and ME throughout the game.
 * @author Clay Davis
 *
 * @param bgmVolume
 * @text Global BGM Volume
 * @type number
 * @min 0
 * @max 100
 * @desc Set the global BGM volume level. Overrides all individual settings.
 * @default 100
 *
 * @param bgsVolume
 * @text Global BGS Volume
 * @type number
 * @min 0
 * @max 100
 * @desc Set the global BGS volume level. Overrides all individual settings.
 * @default 100
 *
 * @param seVolume
 * @text Global SE Volume
 * @type number
 * @min 0
 * @max 100
 * @desc Set the global SE volume level. Overrides all individual settings.
 * @default 100
 *
 * @param meVolume
 * @text Global ME Volume
 * @type number
 * @min 0
 * @max 100
 * @desc Set the global ME volume level. Overrides all individual settings.
 * @default 100
 *
 * @help
 * This plugin enforces global volume levels for BGM, BGS, SE, and ME,
 * ignoring custom volume levels set in event commands or the database.
 *
 * How It Works:
 * - All BGM, BGS, SE, and ME played in the game will automatically use
 *   the global volume levels defined in the plugin parameters.
 * - Custom volume levels in event commands or the database are ignored.
 * - The plugin reapplies the global volume settings on scene transitions.
 */

(() => {
    const parameters = PluginManager.parameters('GlobalVolumeControl');
    const bgmVolume = Number(parameters['bgmVolume'] || 100);
    const bgsVolume = Number(parameters['bgsVolume'] || 100);
    const seVolume = Number(parameters['seVolume'] || 100);
    const meVolume = Number(parameters['meVolume'] || 100);

    // Apply global BGM volume
    const _AudioManager_playBgm = AudioManager.playBgm;
    AudioManager.playBgm = function(bgm, pos) {
        if (bgm) {
            bgm.volume = bgmVolume;
        }
        _AudioManager_playBgm.call(this, bgm, pos);
    };

    // Apply global BGS volume
    const _AudioManager_playBgs = AudioManager.playBgs;
    AudioManager.playBgs = function(bgs, pos) {
        if (bgs) {
            bgs.volume = bgsVolume;
        }
        _AudioManager_playBgs.call(this, bgs, pos);
    };

    // Apply global SE volume
    const _AudioManager_playSe = AudioManager.playSe;
    AudioManager.playSe = function(se) {
        if (se) {
            se.volume = seVolume;
        }
        _AudioManager_playSe.call(this, se);
    };

    // Apply global ME volume
    const _AudioManager_playMe = AudioManager.playMe;
    AudioManager.playMe = function(me) {
        if (me) {
            me.volume = meVolume;
        }
        _AudioManager_playMe.call(this, me);
    };

    // Reapply global volume settings when scenes start
    const _Scene_Base_start = Scene_Base.prototype.start;
    Scene_Base.prototype.start = function() {
        _Scene_Base_start.call(this);
        AudioManager.bgmVolume = bgmVolume;
        AudioManager.bgsVolume = bgsVolume;
        AudioManager.seVolume = seVolume;
        AudioManager.meVolume = meVolume;
    };
})();